<?php
/**
 * Plugin Name: tagDiv AMP
 * Plugin URI: http://forum.tagdiv.com/tagdiv-amp/
 * Description: Add AMP support to your WordPress site.
 * Author: tagDiv
 * Version: 1.3 | built on 06.02.2019 12:12
 * Author URI: http://tagdiv.com
 */

/***
 *  tagDiv AMP is tagDiv's solution for implementing Google AMP in WordPress.
 *
 *  Copyright © 2018 tagDiv
 *
 *  Our portfolio is here: https://themeforest.net/user/tagdiv/portfolio
 *
 *  \--> tagDiv, 2018 <--/
 */

//hash
define('TD_AMP', 'fe5d8f21dc824f170335a24bf11e687d');

// theme plugin version
define( 'TD_AMP__VERSION', '1.2' );

define( 'AMP__FILE__', __FILE__ );
define( 'AMP__DIR__', dirname( __FILE__ ) );
define( 'AMP__VERSION', '0.6.2' );

require_once AMP__DIR__ . '/td_amp_version_check.php';

require_once AMP__DIR__ . '/includes/class-amp-autoloader.php';
AMP_Autoloader::register();

require_once AMP__DIR__ . '/back-compat/back-compat.php';
require_once AMP__DIR__ . '/includes/amp-helper-functions.php';
require_once AMP__DIR__ . '/includes/admin/functions.php';

register_activation_hook( __FILE__, 'amp_activate' );
function amp_activate() {
    amp_after_setup_theme();
    if ( ! did_action( 'amp_init' ) ) {
        amp_init();
    }
    flush_rewrite_rules();
}

register_deactivation_hook( __FILE__, 'amp_deactivate' );
function amp_deactivate() {
    // We need to manually remove the amp endpoint
    global $wp_rewrite;
    foreach ( $wp_rewrite->endpoints as $index => $endpoint ) {
        if ( AMP_QUERY_VAR === $endpoint[1] ) {
            unset( $wp_rewrite->endpoints[ $index ] );
            break;
        }
    }

    flush_rewrite_rules();
}

/**
 * Set up AMP.
 *
 * This function must be invoked through the 'after_setup_theme' action to allow
 * the AMP setting to declare the post types support earlier than plugins/theme.
 *
 * @since 0.6
 */
function amp_after_setup_theme() {

    if ( td_amp_version_check::is_theme_compatible() === false ) {
        //todo - initialization stopped but 'templates/functions.php' is still loaded by the theme
        return;
    }

    if ( false === apply_filters( 'amp_is_enabled', true ) ) {
        return;
    }

    if ( ! defined( 'AMP_QUERY_VAR' ) ) {
        /**
         * Filter the AMP query variable.
         *
         * @since 0.3.2
         * @param string $query_var The AMP query variable.
         */
        define( 'AMP_QUERY_VAR', apply_filters( 'amp_query_var', 'amp' ) );
    }

    add_action( 'init', 'amp_init' );
    add_action( 'admin_init', 'AMP_Options_Manager::register_settings' );
    add_filter( 'amp_post_template_analytics', 'amp_add_custom_analytics' );
    add_action( 'wp_loaded', 'amp_post_meta_box' );
    add_action( 'wp_loaded', 'amp_add_options_menu' );
    add_action( 'parse_query', 'amp_correct_query_when_is_front_page' );
    AMP_Post_Type_Support::add_post_type_support();
}
add_action( 'after_setup_theme', 'amp_after_setup_theme', 5 );

/**
 * Init AMP.
 *
 * @since 0.1
 */
function amp_init() {

    /**
     * Triggers on init when AMP plugin is active.
     *
     * @since 0.3
     */
    do_action( 'amp_init' );

    load_plugin_textdomain( 'amp', false, plugin_basename( AMP__DIR__ ) . '/languages' );

    add_rewrite_endpoint( AMP_QUERY_VAR, EP_PERMALINK );

    add_filter( 'request', 'amp_force_query_var_value' );
    add_action( 'wp', 'amp_maybe_add_actions' );

    // Redirect the old url of amp page to the updated url.
    add_filter( 'old_slug_redirect_url', 'amp_redirect_old_slug_to_new_url' );

    if ( class_exists( 'Jetpack' ) && ! ( defined( 'IS_WPCOM' ) && IS_WPCOM ) ) {
        require_once( AMP__DIR__ . '/jetpack-helper.php' );
    }
}

// Make sure the `amp` query var has an explicit value.
// Avoids issues when filtering the deprecated `query_string` hook.
function amp_force_query_var_value( $query_vars ) {
    if ( isset( $query_vars[ AMP_QUERY_VAR ] ) && '' === $query_vars[ AMP_QUERY_VAR ] ) {
        $query_vars[ AMP_QUERY_VAR ] = 1;
    }
    return $query_vars;
}

/**
 * Fix up WP_Query for front page when amp query var is present.
 *
 * Normally the front page would not get served if a query var is present other than preview, page, paged, and cpage.
 *
 * @since 0.6
 * @see WP_Query::parse_query()
 * @link https://github.com/WordPress/wordpress-develop/blob/0baa8ae85c670d338e78e408f8d6e301c6410c86/src/wp-includes/class-wp-query.php#L951-L971
 *
 * @param WP_Query $query Query.
 */
function amp_correct_query_when_is_front_page( WP_Query $query ) {
    $is_front_page_query = (
        $query->is_main_query()
        &&
        $query->is_home()
        &&
        // Is AMP endpoint.
        false !== $query->get( AMP_QUERY_VAR, false )
        &&
        // Is query not yet fixed uo up to be front page.
        ! $query->is_front_page()
        &&
        // Is showing pages on front.
        'page' === get_option( 'show_on_front' )
        &&
        // Has page on front set.
        get_option( 'page_on_front' )
        &&
        // See line in WP_Query::parse_query() at <https://github.com/WordPress/wordpress-develop/blob/0baa8ae/src/wp-includes/class-wp-query.php#L961>.
        0 === count( array_diff( array_keys( wp_parse_args( $query->query ) ), array( AMP_QUERY_VAR, 'preview', 'page', 'paged', 'cpage' ) ) )
    );
    if ( $is_front_page_query ) {
        $query->is_home     = false;
        $query->is_page     = true;
        $query->is_singular = true;
        $query->set( 'page_id', get_option( 'page_on_front' ) );
    }
}

/**
 * Add AMP actions when the request can be served as AMP.
 *
 * Actions will only be added if the request is for a singular post (including front page and page for posts), excluding feeds.
 *
 * @since 0.2
 */
function amp_maybe_add_actions() {
    global $wp_query;
    if ( ! ( is_singular() || $wp_query->is_posts_page ) || is_feed() ) {
        return;
    }

    $is_amp_endpoint = is_amp_endpoint();

    /**
     * Queried post object.
     *
     * @var WP_Post $post
     */
    $post = get_queried_object();


    if ( ! post_supports_amp( $post ) ) {
        if ( $is_amp_endpoint ) {
            wp_safe_redirect( get_permalink( $post->ID ), 302 ); // Temporary redirect because AMP may be supported in future.
            exit;
        }
        return;
    }

    if ( $is_amp_endpoint ) {
        amp_prepare_render();
    } else {
        amp_add_frontend_actions();
    }
}

function amp_load_classes() {
    _deprecated_function( __FUNCTION__, '0.6' );
}

function amp_add_frontend_actions() {
    require_once AMP__DIR__ . '/includes/amp-frontend-actions.php';
}

function amp_add_post_template_actions() {
    require_once AMP__DIR__ . '/includes/amp-post-template-actions.php';
    require_once AMP__DIR__ . '/includes/amp-post-template-functions.php';
    amp_post_template_init_hooks();
}

function amp_prepare_render() {
    add_action( 'template_redirect', 'amp_render' );
}

/**
 * Render AMP for queried post.
 *
 * @since 0.1
 */
function amp_render() {
    // Note that queried object is used instead of the ID so that the_preview for the queried post can apply.
    $post = get_queried_object();
    if ( $post instanceof WP_Post ) {
        amp_render_post( $post );
        exit;
    }
}

/**
 * Render AMP post template.
 *
 * @since 0.5
 * @param WP_Post|int $post Post.
 * @global WP_Query $wp_query
 */
function amp_render_post( $post ) {
    global $wp_query;

    if ( ! ( $post instanceof WP_Post ) ) {
        $post = get_post( $post );
        if ( ! $post ) {
            return;
        }
    }
    $post_id = $post->ID;

    /*
     * If amp_render_post is called directly outside of the standard endpoint, is_amp_endpoint() will return false,
     * which is not ideal for any code that expects to run in an AMP context.
     * Let's force the value to be true while we render AMP.
     */
    $was_set = isset( $wp_query->query_vars[ AMP_QUERY_VAR ] );
    if ( ! $was_set ) {
        $wp_query->query_vars[ AMP_QUERY_VAR ] = true;
    }

    /**
     * Fires before rendering a post in AMP.
     *
     * @since 0.2
     *
     * @param int $post_id Post ID.
     */
    do_action( 'pre_amp_render_post', $post_id );

    amp_add_post_template_actions();
    $template = new AMP_Post_Template( $post );
    $template->load();

    if ( ! $was_set ) {
        unset( $wp_query->query_vars[ AMP_QUERY_VAR ] );
    }
}

/**
 * Bootstraps the AMP customizer.
 *
 * @since 0.4
 */
function _amp_bootstrap_customizer() {
    add_action( 'after_setup_theme', 'amp_init_customizer' );
}
add_action( 'plugins_loaded', '_amp_bootstrap_customizer', 9 ); // Should be hooked before priority 10 on 'plugins_loaded' to properly unhook core panels.

/**
 * Redirects the old AMP URL to the new AMP URL.
 * If post slug is updated the amp page with old post slug will be redirected to the updated url.
 *
 * @param  string $link New URL of the post.
 *
 * @return string $link URL to be redirected.
 */
function amp_redirect_old_slug_to_new_url( $link ) {

    if ( is_amp_endpoint() ) {
        $link = trailingslashit( trailingslashit( $link ) . AMP_QUERY_VAR );
    }

    return $link;
}

